// Memory client that interfaces with MemManager
public class MemClient {
	private Handle[] recordArray;		// Client's array of handles
	private MemManager memoryManager;	// Instance of a memory manager
	
	public MemClient(int input_numRecs, int input_poolSize) {
		// Initialize the record array and create a new memory manager
		recordArray = new Handle[input_numRecs];
		memoryManager = new MemManager(input_poolSize);
	}
	
	public void processLine(String line) {
		// Split the passed line into tokens
		String delim = "[ ]+";
		String[] tokens = line.split(delim);
		
		// Parse tokens to determine what action to take
		if (tokens[0].compareToIgnoreCase("insert") == 0) {
			processInsertCommand(tokens, recordArray, memoryManager);
		} else if (tokens[0].compareToIgnoreCase("remove") == 0) {
			processRemoveCommand(tokens, recordArray, memoryManager);
		} else if (tokens[0].compareToIgnoreCase("print") == 0) {
			processPrintCommand(tokens, recordArray, memoryManager);
		} else {
			System.out.printf("\"%s\" command is not supported\n", tokens[0]);
		}
	}
	
	public void processInsertCommand(String[] tokens, Handle[] recordArray, MemManager memoryManager) {
		// Process insert command
		if (tokens.length != 5) {
			System.out.printf("Arguments must be in format \"insert recnum x y name\"\n");
		} else {
			try {
				// Check for array bounds for record array
				int recordID = Integer.parseInt(tokens[1]);
				if (recordID < 0 || recordID > recordArray.length - 1) {
					System.out.printf("Arguments out of bound\n");
				} else {
					// Delete record in record array if it exists
					if (recordArray[recordID] != null) {
						memoryManager.remove(recordArray[recordID]);
						recordArray[recordID] = null;
					}
					// Serialize to byte array and send to memory manager
					int xpos = Integer.parseInt(tokens[2]);
					int ypos = Integer.parseInt(tokens[3]);
					byte[] buffer = ByteStringConverter.convertToByteArray(xpos, ypos, tokens[4]);
					Handle retHandle = memoryManager.insert(buffer, buffer.length);
					if (retHandle == null) {
						System.out.printf("No free space is avaliable to store the record\n");
					} else {
						// Save returned handle to the record array
						recordArray[recordID] = retHandle;
					}
				}
			} catch (NumberFormatException e) {
				System.out.printf("Error attempting to parse int values\n");
			}
		}
	}
	
	public void processRemoveCommand(String[] tokens, Handle[] recordArray, MemManager memoryManager) {
		// Process remove command
		if (tokens.length != 2) {
			System.out.printf("A record ID must be supplied\n");
		} else {
			try {
				int recordID = Integer.parseInt(tokens[1]);
				if (recordID < 0 || recordID > recordArray.length - 1) {
					System.out.printf("Arguments out of bound\n");
				} else {
					// Remove record from record array if it exists
					if (recordArray[recordID] != null) {
						memoryManager.remove(recordArray[recordID]);
						recordArray[recordID] = null;
					} else {
						System.out.printf("No record found for record ID %d\n", recordID);
					}
				}
			} catch (NumberFormatException e) {
				System.out.printf("Error attempting to parse int values\n");
			}
		}
	}
	
	public void processPrintCommand(String[] tokens, Handle[] recordArray, MemManager memoryManager) {
		byte[] fromPool = new byte[255];
		// Process print command
		if (tokens.length == 1) {
			// Print out every entry in recordArray
			for (int i = 0; i < recordArray.length; i++) {
				if (recordArray[i] != null) {
					fromPool = new byte[255];
					System.out.printf("Record #%d at address %d: ", i, recordArray[i].getAddress());
					memoryManager.get(fromPool, recordArray[i], 255);
					String[] strArray = ByteStringConverter.convertToStringArray(fromPool);
					System.out.printf("%s (%s,%s)\n", strArray[2].trim(), strArray[0], strArray[1]);
				}
			}
			// Print out free block list
			memoryManager.dump();
		} else if (tokens.length == 2) {
			// Print out specified record
			try {
				int recordID = Integer.parseInt(tokens[1]);
				if (recordID < 0 || recordID > recordArray.length - 1) {
					System.out.printf("Arguments out of bound\n");
				} else {
					if (recordArray[recordID] != null) {
						// Get and print out record
						memoryManager.get(fromPool, recordArray[recordID], 255);
						String[] strArray = ByteStringConverter.convertToStringArray(fromPool);
						System.out.printf("Record #%d: %s (%s,%s)\n", recordID, strArray[2].trim(), strArray[0], strArray[1]);
					} else {
						System.out.printf("No record found for record ID %d\n", recordID);
					}
				}
			} catch (NumberFormatException e) {
				System.out.printf("Error attempting to parse int values\n");
			}
		} else {
			System.out.printf("Invalud number of arguments\n");
		}
	}
}
